<?php

namespace App\Controllers;

use App\Libraries\CIAuth;
use App\Libraries\InputCleaner;
use App\Libraries\PasswordVerify;

use App\Models\CIAuthModel;
use App\Models\PanelModel;
use App\Models\UsersModel;

class UsersController extends BaseController {
    
    public function __construct(){
        $panelModel = new PanelModel();
        $this->settings = $panelModel->getSettings();
        $this->details = new UsersModel();
    }

    public function index(){
        if(!CIAuth::check()){
            return redirect()->to(base_url('ns-admin/login'));
        }
        
        $data = [
            'pageTitle' => 'Manage Users',
            'currentFile' => 'users'
        ];
        
        $data['settings'] = $this->settings;
        
        // Get search keyword
        $keyword = $this->request->getGet('keyword') ?? '';
        
        if (!empty($keyword)) {
            // If searching, show all results without pagination
            $data['result'] = $this->details
                ->like('user_name', $keyword)
                ->orderBy('id', 'DESC')
                ->findAll(); // No pagination here
                
            // Disable pagination variables
            $data['total_pages'] = 1; 
            $data['page'] = 1;
            $data['targetpage'] = '';
        } else {
            // Apply pagination only when not searching
            $limit = 12; // Number of items per page
            $page = $this->request->getGet('page') ?? 1;
            $page = max(1, intval($page));
            $offset = ($page - 1) * $limit;
            
            // Get total count only for non-search mode
            $total_records = $this->details->countAllResults();
            $total_pages = ($total_records > 0) ? ceil($total_records / $limit) : 1;
            
            $data['result'] = $this->details
                ->orderBy('id', 'DESC')
                ->findAll($limit, $offset);
                
            // Pass pagination data to view
            $data['total_pages'] = $total_pages;
            $data['page'] = $page;
            $data['targetpage'] = base_url('ns-admin/manage-users');
        }

        return view('manage_users', $data);
    }
    
    public function create(){
        if(!CIAuth::check()){
            return redirect()->to(base_url('ns-admin/login'));
        }
        
        $data = [
            'pageTitle' => 'Create User',
            'currentFile' => 'users',
            'pageSave' => 'Create'
        ];
        $data['settings'] = $this->settings;
        return view('create_user', $data);
    }
    
    public function edit($id = null){
        if(!CIAuth::check()){
            return redirect()->to(base_url('ns-admin/login'));
        }
        
        $dataDetails = $this->details->find($id);
        
        $data = [
            'pageTitle' => 'Edit User',
            'currentFile' => 'users',
            'pageSave' => 'Save',
            'user_id' => $dataDetails['id'],
            'row' => $dataDetails,
        ];
        $data['settings'] = $this->settings;
        return view('create_user', $data);
    }
    
    public function createHandler(){
        if(!CIAuth::check()){
            return redirect()->to(base_url('ns-admin/login'));
        }
        
        $editID = $this->request->getPost('user_id');
        
        // Simple CSRF check with better error handling
        $postedToken = $this->request->getPost('csrf_test_name');
        if (empty($postedToken)) {
            $message = array('message' => 'CSRF token is missing. Please refresh the page and try again.', 'class' => 'error');
            session()->set('response_msg', $message);
            if(empty($editID)){
                return redirect()->to(base_url('ns-admin/create-users'));
            } else {
                return redirect()->to(base_url('ns-admin/create-users/'.$editID));
            }
        }
        if ($postedToken !== csrf_hash()) {
            $message = array('message' => 'Security token validation failed. Please refresh the page and try again.', 'class' => 'error');
            session()->set('response_msg', $message);
            if(empty($editID)){
                return redirect()->to(base_url('ns-admin/create-users'));
            } else {
                return redirect()->to(base_url('ns-admin/create-users/'.$editID));
            }
        }
        
        if(empty($editID)){
            
            $imageName = $this->handleImageUpload();
            
            $hashedPassword = PasswordVerify::hash(trim($this->request->getPost('user_password')));
            $data = array(
                'user_type'=>'Normal',											 
                'user_name'  => $this->request->getPost('user_name'),				    
                'user_email'  =>  $this->request->getPost('user_email'),
                'user_password'  =>  $hashedPassword,
                'user_phone'  =>  $this->request->getPost('user_phone'),
                'user_gender'  =>  $this->request->getPost('user_gender'),
                'registered_on'  =>  strtotime(date('d-m-Y h:i:s A')),
                'profile_img'  => $imageName,
                'status'  =>  '1'
            );
            $this->details->insert($data);
            
            $message = array('message' => lang('Validation.added'),'class' => 'success');
            session()->set('response_msg', $message);
            return redirect()->to(base_url('ns-admin/manage-users'));
            
        } else {
            
            $dataDetails = $this->details->find($editID);
            if($dataDetails){
                if ($this->request->getPost('submit') !== null) {
                    $imageName = $this->handleImageUpload($dataDetails['profile_img']);
                    if($this->request->getPost('user_password') != ""){
                        $hashedPassword = PasswordVerify::hash(trim($this->request->getPost('user_password')));
                        $data = array(
                            'user_name'  => $this->request->getPost('user_name'),				    
                            'user_email'  =>  $this->request->getPost('user_email'),
                            'user_password'  =>  $hashedPassword,
                            'user_phone'  =>  $this->request->getPost('user_phone'),
                            'user_gender'  =>  $this->request->getPost('user_gender'),
                            'profile_img'  => $imageName
                        ); 
                    } else {
                        $data = array(
                            'user_name'  => $this->request->getPost('user_name'),				    
                            'user_email'  =>  $this->request->getPost('user_email'),
                            'user_phone'  =>  $this->request->getPost('user_phone'),
                            'user_gender'  =>  $this->request->getPost('user_gender'),
                            'profile_img'  => $imageName
                        ); 
                    }
                    
                    $this->details->update($editID, $data);
                    $message = array('message' => lang('Validation.updated'),'class' => 'success');
                } else {
                    $message = array('message' => lang('Validation.updated_failed'),'class' => 'error');
                }
            } else {
                $message = array('message' => lang('Validation.updated_failed_user'),'class' => 'error');
            }
            session()->set('response_msg', $message);
            return redirect()->to(base_url('ns-admin/create-user/'.$editID));
        }
    }
    
    public function delete($id = null){
        // Check if the user is authorized to perform the action
        if (!CIAuth::check()) {
            return $this->response->setJSON(['status' => 0, 'message' => 'Unauthorized']);
        }
        
        $allowedAdminTypes = [1, 3];
        if(!in_array(CIAuth::adminType(), $allowedAdminTypes)){
            return $this->response->setJSON(['status' => 0, 'message' => 'Unauthorized']);
        }
        
        // Validate ID
        if (is_null($id) || !is_numeric($id)) {
            return $this->response->setJSON(['status' => 0, 'message' => 'Invalid ID']);
        }
        
        // Check if the record exists
        $record = $this->details->find($id);
        if (!$record) {
            return $this->response->setJSON(['status' => 0, 'message' => 'Record not found']);
        }
        
        try {
            // Attempt to delete the record
            if ($this->details->delete($id)) {
                
                $imageDirectory = 'images/';
                $existingImage = $record['profile_img'];
                if ($existingImage && file_exists($imageDirectory . $existingImage)) {
                    unlink($imageDirectory . $existingImage);
                }
                
                session()->set('response_msg', ['message' => lang('Validation.deleted'), 'class' => 'success']);
                return $this->response->setJSON(['status' => 1, 'message' => 'Delete successful']);
            } else {
                return $this->response->setJSON(['status' => 0, 'message' => 'Failed to delete']);
            }
        } catch (\Exception $e) {
            // Log and handle potential errors during deletion
            log_message('error', 'Error deleting record: ' . $e->getMessage());
            return $this->response->setJSON(['status' => 0, 'message' => 'An error occurred while deleting']);
        }
    }
    
    public function status($id = null){
        // Ensure user is authenticated
        if (!CIAuth::check()) {
            return $this->response->setStatusCode(401)
                ->setJSON(['status' => 0, 'msg' => 'Unauthorized']);
        }
        
        $allowedAdminTypes = [1, 3];
        if(!in_array(CIAuth::adminType(), $allowedAdminTypes)){
            return $this->response->setStatusCode(401)
                ->setJSON(['status' => 0, 'msg' => 'Unauthorized']);
        }
        
        // Validate ID
        if (is_null($id) || !is_numeric($id)) {
            return $this->response->setJSON(['status' => 0, 'msg' => 'Invalid ID']);
        }
        
        // Check if record exists
        $record = $this->details->find($id);
        if (!$record) {
            return $this->response->setJSON(['status' => 0, 'msg' => 'Record not found']);
        }
        
        // Ensure the request is a POST request
        if (!$this->request->is('post')) {
            return $this->response->setStatusCode(405)
                ->setJSON(['status' => 0, 'msg' => 'Method Not Allowed']);
        }
        
        try {
            $for_action = $this->request->getPost('for_action');
            $column = $this->request->getPost('column');
            
            // Update data
            $data = [$column => ($for_action === 'enable' ? '1' : '0')];
            $message = ($for_action === 'enable') ? lang('Validation.enabled') : lang('Validation.disabled');
            
            $this->details->update($id, $data);
            
            return $this->response->setJSON([
                'status' => 1,
                'action' => $for_action,
                'msg' => $message,
                'class' => 'success'
            ]);
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'status' => 0,
                'msg' => 'An error occurred while updating the record'
            ]);
        }
    }
    
    private function handleImageUpload($existingImage = '') {
        $suffix = '_users';
        $file = $this->request->getFile('image_add');
        
        // Check if the file exists
        if (!$file) {
            return $existingImage; // Return the existing image if no file was uploaded
        }
        
        if ($file->isValid() && !$file->hasMoved()) {
            
            $imageDirectory = 'images/';
            
            // Check if the directory exists; if not, create it
            if (!is_dir($imageDirectory)) {
                mkdir($imageDirectory, 0755, true); // Create directory with safer permissions
            }
            
            // Delete existing image if it exists
            if ($existingImage && file_exists($imageDirectory . $existingImage)) {
                unlink($imageDirectory . $existingImage);
            }
            
            // Generate new image name with a suffix
            $extension = $file->getExtension(); // Get file extension
            $imageName = rand(0, 99999) . $suffix . '.' . $extension; // Append suffix
            
            // Move the file with the new name
            $file->move($imageDirectory, $imageName);
            
            return $imageName;
        }
        return $existingImage;
    }
}